<?php
class TokenMonitor {
    private $db;
    private $telegram_token = '7548610464:AAEHqe-mvHNCpmpONIbeD6qLovwH3UKzKTU';
    private $telegram_bot_username = 'ICT_ForexTradingSignals_bot';
    private $chat_id; // باید تنظیم شود

    public function __construct($db_config) {
        $this->db = new PDO(
            "mysql:host={$db_config['host']};dbname={$db_config['dbname']}", 
            $db_config['user'], 
            $db_config['pass']
        );
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }

    // مانیتورینگ توکن‌های جدید در پینک‌سیل
    public function monitorPinksale() {
        $pinksale_data = $this->fetchPinksaleData();
        foreach ($pinksale_data as $token) {
            if (!$this->tokenExists($token['address'])) {
                $this->addNewToken($token);
                $this->analyzeToken($token);
            }
        }
    }

    // دریافت اطلاعات از پینک‌سیل
    private function fetchPinksaleData() {
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.pinksale.finance/api/v1/solana/launchpads",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                "Accept: application/json",
                "User-Agent: Mozilla/5.0"
            ],
        ]);
        $response = curl_exec($curl);
        curl_close($curl);
        return json_decode($response, true);
    }

    // تحلیل تکنیکال
    private function analyzeTechnicalIndicators($token_id) {
        $data = $this->getPriceHistory($token_id);
        
        // محاسبه میانگین متحرک
        $sma20 = $this->calculateSMA($data, 20);
        $sma50 = $this->calculateSMA($data, 50);
        
        // بررسی RSI
        $rsi = $this->calculateRSI($data);
        
        // تحلیل روند
        $trend = $this->analyzeTrend($data);
        
        return [
            'sma20' => $sma20,
            'sma50' => $sma50,
            'rsi' => $rsi,
            'trend' => $trend
        ];
    }

    // محاسبه میانگین متحرک
    private function calculateSMA($data, $period) {
        $prices = array_column($data, 'close_price');
        $sma = [];
        
        for ($i = $period - 1; $i < count($prices); $i++) {
            $sum = 0;
            for ($j = 0; $j < $period; $j++) {
                $sum += $prices[$i - $j];
            }
            $sma[] = $sum / $period;
        }
        
        return $sma;
    }

    // محاسبه RSI
    private function calculateRSI($data, $period = 14) {
        $prices = array_column($data, 'close_price');
        $gains = [];
        $losses = [];
        
        // محاسبه تغییرات قیمت
        for ($i = 1; $i < count($prices); $i++) {
            $change = $prices[$i] - $prices[$i - 1];
            if ($change >= 0) {
                $gains[] = $change;
                $losses[] = 0;
            } else {
                $gains[] = 0;
                $losses[] = abs($change);
            }
        }
        
        // محاسبه میانگین gains و losses
        $avg_gain = array_sum(array_slice($gains, 0, $period)) / $period;
        $avg_loss = array_sum(array_slice($losses, 0, $period)) / $period;
        
        // محاسبه RS و RSI
        $rs = $avg_gain / ($avg_loss ?: 1); // جلوگیری از تقسیم بر صفر
        $rsi = 100 - (100 / (1 + $rs));
        
        return $rsi;
    }

    // تحلیل روند قیمت
    private function analyzeTrend($data) {
        $prices = array_column($data, 'close_price');
        $highs = array_column($data, 'high_price');
        $lows = array_column($data, 'low_price');
        
        // بررسی روند در بازه‌های مختلف
        $trend_daily = $this->determineTrend($prices, 24); // 24 ساعت
        $trend_weekly = $this->determineTrend($prices, 168); // یک هفته
        
        return [
            'daily' => $trend_daily,
            'weekly' => $trend_weekly,
            'higher_highs' => $this->checkHigherHighs($highs),
            'higher_lows' => $this->checkHigherLows($lows)
        ];
    }

    // تعیین جهت روند
    private function determineTrend($prices, $period) {
        $recent_prices = array_slice($prices, -$period);
        $linear_regression = $this->calculateLinearRegression($recent_prices);
        
        if ($linear_regression['slope'] > 0.01) {
            return 'uptrend';
        } elseif ($linear_regression['slope'] < -0.01) {
            return 'downtrend';
        } else {
            return 'sideways';
        }
    }

    // محاسبه رگرسیون خطی برای تشخیص روند
    private function calculateLinearRegression($prices) {
        $x = range(1, count($prices));
        $y = $prices;
        
        $n = count($prices);
        $sum_x = array_sum($x);
        $sum_y = array_sum($y);
        $sum_xy = 0;
        $sum_xx = 0;
        
        for ($i = 0; $i < $n; $i++) {
            $sum_xy += ($x[$i] * $y[$i]);
            $sum_xx += ($x[$i] * $x[$i]);
        }
        
        $slope = ($n * $sum_xy - $sum_x * $sum_y) / ($n * $sum_xx - $sum_x * $sum_x);
        $intercept = ($sum_y - $slope * $sum_x) / $n;
        
        return [
            'slope' => $slope,
            'intercept' => $intercept
        ];
    }

    // ارسال هشدار به تلگرام
    public function sendTelegramAlert($message) {
        $url = "https://api.telegram.org/bot{$this->telegram_token}/sendMessage";
        $params = [
            'chat_id' => $this->chat_id,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        return json_decode($response, true);
    }

    // امتیازدهی به توکن
    private function scoreToken($token_data) {
        $score = 0;
        $weights = [
            'liquidity' => 0.25,
            'social' => 0.20,
            'technical' => 0.25,
            'fundamentals' => 0.30
        ];
        
        // امتیاز لیکوئیدیتی
        $liquidity_score = $this->scoreLiquidity($token_data['liquidity']);
        
        // امتیاز سوشال مدیا
        $social_score = $this->scoreSocialMetrics($token_data['social']);
        
        // امتیاز تکنیکال
        $technical_score = $this->scoreTechnicalMetrics($token_data);
        
        // امتیاز فاندامنتال
        $fundamental_score = $this->scoreFundamentals($token_data);
        
        // محاسبه امتیاز نهایی
        $final_score = ($liquidity_score * $weights['liquidity']) +
                      ($social_score * $weights['social']) +
                      ($technical_score * $weights['technical']) +
                      ($fundamental_score * $weights['fundamentals']);
                      
        return $final_score;
    }

    // ذخیره امتیاز در دیتابیس
    private function saveScore($token_id, $score_data) {
        $stmt = $this->db->prepare("
            INSERT INTO token_scores 
            (token_id, liquidity_score, social_score, developer_score, total_score, timestamp)
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        
        return $stmt->execute([
            $token_id,
            $score_data['liquidity_score'],
            $score_data['social_score'],
            $score_data['developer_score'],
            $score_data['total_score']
        ]);
    }
    
    // بررسی ریسک‌های احتمالی
    private function assessRisks($token_data) {
        $risks = [];
        
        // بررسی لیکوئیدیتی کم
        if ($token_data['liquidity'] < 50000) {
            $risks[] = [
                'type' => 'low_liquidity',
                'severity' => 'high',
                'description' => 'Liquidity is below $50,000'
            ];
        }
        
        // بررسی تمرکز هولدرها
        if ($this->checkHolderConcentration($token_data['token_address'])) {
            $risks[] = [
                'type' => 'holder_concentration',
                'severity' => 'high',
                'description' => 'High concentration of tokens in top wallets'
            ];
        }
        
        // بررسی حجم معاملات کم
        if ($token_data['volume_24h'] < 10000) {
            $risks[] = [
                'type' => 'low_volume',
                'severity' => 'medium',
                'description' => '24h trading volume is below $10,000'
            ];
        }
        
        // بررسی سن توکن
        if ($this->isNewToken($token_data['launch_date'])) {
            $risks[] = [
                'type' => 'new_token',
                'severity' => 'medium',
                'description' => 'Token is less than 24 hours old'
            ];
        }
        
        return $risks;
    }

    // بررسی تمرکز هولدرها
    private function checkHolderConcentration($token_address) {
        $holders = $this->getTopHolders($token_address);
        $total_supply = $this->getTokenSupply($token_address);
        
        $top_10_percentage = 0;
        foreach (array_slice($holders, 0, 10) as $holder) {
            $top_10_percentage += ($holder['balance'] / $total_supply) * 100;
        }
        
        return $top_10_percentage > $this->config['monitor']['max_holder_concentration'];
    }

    // ایجاد سیگنال خرید
    private function generateBuySignal($token_data, $analysis) {
        if ($analysis['score'] >= $this->config['scoring']['thresholds']['buy_signal']) {
            $message = $this->formatBuySignal($token_data, $analysis);
            $this->sendTelegramAlert($message);
            
            // ذخیره سیگنال در دیتابیس
            $this->saveBuySignal($token_data['token_address'], $analysis);
        }
    }

    // فرمت‌بندی پیام سیگنال خرید
    private function formatBuySignal($token_data, $analysis) {
        return "🚀 *سیگنال خرید جدید*\n\n" .
               "نام توکن: {$token_data['name']}\n" .
               "نماد: {$token_data['symbol']}\n" .
               "قیمت: \${$token_data['price_usd']}\n" .
               "لیکوئیدیتی: \${$token_data['liquidity']}\n" .
               "حجم 24h: \${$token_data['volume_24h']}\n\n" .
               "📊 *تحلیل تکنیکال*\n" .
               "روند روزانه: {$analysis['trend']['daily']}\n" .
               "RSI: {$analysis['rsi']}\n\n" .
               "⭐️ امتیاز کل: {$analysis['score']}/100\n\n" .
               "🔗 [مشاهده در دکس‌اسکرینر](https://dexscreener.com/solana/{$token_data['token_address']})";
    }

    // بررسی امنیت قرارداد هوشمند
    private function checkContractSecurity($token_address) {
        // TODO: اضافه کردن بررسی‌های امنیتی بیشتر
        $security_checks = [
            'is_proxy' => $this->isProxyContract($token_address),
            'has_mint_function' => $this->hasMintFunction($token_address),
            'is_verified' => $this->isContractVerified($token_address),
            'audit_reports' => $this->getAuditReports($token_address)
        ];
        
        return $security_checks;
    }

    // ذخیره تحلیل‌ها در دیتابیس
    private function saveAnalysis($token_id, $analysis_data) {
        $stmt = $this->db->prepare("
            INSERT INTO token_analysis 
            (token_id, technical_data, risk_assessment, security_checks, timestamp)
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        return $stmt->execute([
            $token_id,
            json_encode($analysis_data['technical']),
            json_encode($analysis_data['risks']),
            json_encode($analysis_data['security'])
        ]);
    }
}