<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../src/TokenMonitor.php';
require_once __DIR__ . '/../src/PriceAnalyzer.php';
require_once __DIR__ . '/../src/RiskAnalyzer.php';
require_once __DIR__ . '/../src/SocialAnalyzer.php';
require_once __DIR__ . '/../src/TelegramBot.php';

use TokenMonitor\TokenMonitor;
use TokenMonitor\PriceAnalyzer;
use TokenMonitor\RiskAnalyzer;
use TokenMonitor\SocialAnalyzer;
use TokenMonitor\TelegramBot;

// تنظیم timezone
date_default_timezone_set('UTC');

class Monitor {
    private $db;
    private $config;
    private $token_monitor;
    private $price_analyzer;
    private $risk_analyzer;
    private $social_analyzer;
    private $telegram_bot;
    private $log_file;

    public function __construct() {
        $this->log_file = __DIR__ . '/../logs/monitor_' . date('Y-m-d') . '.log';
        $this->initializeLogger();
        
        try {
            $this->loadConfig();
            $this->connectDatabase();
            $this->initializeComponents();
        } catch (\Exception $e) {
            $this->log("CRITICAL ERROR: " . $e->getMessage());
            die("Initialization failed: " . $e->getMessage());
        }
    }

    // راه‌اندازی لاگر
    private function initializeLogger() {
        if (!file_exists(dirname($this->log_file))) {
            mkdir(dirname($this->log_file), 0777, true);
        }
    }

    // لود کردن کانفیگ
    private function loadConfig() {
        $config_file = __DIR__ . '/../config/config.php';
        if (!file_exists($config_file)) {
            throw new \Exception("Config file not found!");
        }
        $this->config = require $config_file;
    }

    // اتصال به دیتابیس
    private function connectDatabase() {
        try {
            $this->db = new PDO(
                "mysql:host={$this->config['database']['host']};dbname={$this->config['database']['dbname']};charset=utf8mb4",
                $this->config['database']['user'],
                $this->config['database']['pass'],
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (\PDOException $e) {
            throw new \Exception("Database connection failed: " . $e->getMessage());
        }
    }

    // راه‌اندازی کامپوننت‌ها
    private function initializeComponents() {
        $this->token_monitor = new TokenMonitor($this->db, $this->config);
        $this->price_analyzer = new PriceAnalyzer($this->db, $this->config);
        $this->risk_analyzer = new RiskAnalyzer($this->db, $this->config);
        $this->social_analyzer = new SocialAnalyzer($this->db, $this->config);
        $this->telegram_bot = new TelegramBot($this->config, $this->db);
    }

    // اجرای مانیتورینگ
    public function run() {
        $this->log("Starting monitoring process...");

        try {
            // بررسی توکن‌های جدید در پینک‌سیل
            $this->monitorPinksaleTokens();

            // بررسی توکن‌های موجود در دیتابیس
            $this->monitorExistingTokens();

            // ارسال تحلیل روزانه (فقط یک بار در روز)
            if ($this->shouldSendDailyAnalysis()) {
                $this->sendDailyAnalysis();
            }

            $this->log("Monitoring process completed successfully.");
        } catch (\Exception $e) {
            $this->log("ERROR during monitoring: " . $e->getMessage());
            // ارسال نوتیفیکیشن خطا به ادمین
            $this->telegram_bot->sendMessage(
                "❌ Error in monitoring process: " . $e->getMessage(),
                $this->config['admin_chat_id']
            );
        }
    }

    // مانیتورینگ توکن‌های پینک‌سیل
    private function monitorPinksaleTokens() {
        $this->log("Checking Pinksale for new tokens...");
        
        $new_tokens = $this->token_monitor->scanPinksale();
        foreach ($new_tokens as $token) {
            try {
                // تحلیل جامع توکن جدید
                $analysis = $this->analyzeToken($token);
                
                // اگر امتیاز کافی داشت، سیگنال بده
                if ($analysis['scores']['total'] >= $this->config['min_signal_score']) {
                    $this->telegram_bot->sendBuySignal($token, $analysis);
                    $this->log("Buy signal sent for token: {$token['symbol']}");
                }

                // اگر ریسک بالا داشت، هشدار بده
                if (!empty($analysis['risks'])) {
                    $this->telegram_bot->sendRiskAlert($token, $analysis['risks']);
                    $this->log("Risk alert sent for token: {$token['symbol']}");
                }
            } catch (\Exception $e) {
                $this->log("Error analyzing token {$token['symbol']}: " . $e->getMessage());
            }
        }
    }

    // مانیتورینگ توکن‌های موجود
    private function monitorExistingTokens() {
        $this->log("Monitoring existing tokens...");
        
        $tokens = $this->getActiveTokens();
        foreach ($tokens as $token) {
            try {
                // به‌روزرسانی اطلاعات توکن
                $updated_data = $this->updateTokenData($token);
                
                // بررسی تغییرات مهم
                $changes = $this->detectSignificantChanges($token, $updated_data);
                if (!empty($changes)) {
                    $this->notifyChanges($token, $changes);
                }

                // بررسی سیگنال‌های تکنیکال
                $technical_signals = $this->price_analyzer->analyzePriceAction($token['token_address']);
                if ($technical_signals['signals']) {
                    $this->notifyTechnicalSignals($token, $technical_signals);
                }

                // بررسی ریسک‌های جدید
                $new_risks = $this->risk_analyzer->analyzeRisks($updated_data);
                if ($new_risks['risks']) {
                    $this->telegram_bot->sendRiskAlert($token, $new_risks['risks']);
                }
            } catch (\Exception $e) {
                $this->log("Error monitoring token {$token['symbol']}: " . $e->getMessage());
            }
        }
    }

    // دریافت توکن‌های فعال از دیتابیس
    private function getActiveTokens() {
        $stmt = $this->db->prepare("
            SELECT * FROM tokens 
            WHERE status = 'active' 
            AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // به‌روزرسانی اطلاعات توکن
    private function updateTokenData($token) {
        $updated_data = $this->token_monitor->fetchTokenData($token['token_address']);
        
        // به‌روزرسانی در دیتابیس
        $stmt = $this->db->prepare("
            UPDATE tokens SET 
            price = ?, 
            market_cap = ?, 
            volume_24h = ?, 
            holders_count = ?,
            updated_at = NOW()
            WHERE token_address = ?
        ");
        
        $stmt->execute([
            $updated_data['price'],
            $updated_data['market_cap'],
            $updated_data['volume_24h'],
            $updated_data['holders_count'],
            $token['token_address']
        ]);

        return $updated_data;
    }

    // تشخیص تغییرات مهم
    private function detectSignificantChanges($old_data, $new_data) {
        $changes = [];

        // تغییر قیمت بیش از 20%
        $price_change = ($new_data['price'] - $old_data['price']) / $old_data['price'] * 100;
        if (abs($price_change) >= 20) {
            $changes[] = [
                'type' => 'price',
                'change' => $price_change,
                'description' => "Price changed by " . number_format($price_change, 2) . "%"
            ];
        }

        // افزایش حجم معاملات بیش از 200%
        $volume_change = ($new_data['volume_24h'] - $old_data['volume_24h']) / $old_data['volume_24h'] * 100;
        if ($volume_change >= 200) {
            $changes[] = [
                'type' => 'volume',
                'change' => $volume_change,
                'description' => "Volume increased by " . number_format($volume_change, 2) . "%"
            ];
        }

        // تغییر تعداد هولدرها بیش از 50%
        $holders_change = ($new_data['holders_count'] - $old_data['holders_count']) / $old_data['holders_count'] * 100;
        if (abs($holders_change) >= 50) {
            $changes[] = [
                'type' => 'holders',
                'change' => $holders_change,
                'description' => "Holders changed by " . number_format($holders_change, 2) . "%"
            ];
        }

        return $changes;
    }

    // نوتیفیکیشن تغییرات
    private function notifyChanges($token, $changes) {
        $message = "⚡️ <b>تغییرات مهم در {$token['symbol']}</b>\n\n";
        
        foreach ($changes as $change) {
            $emoji = $this->getChangeEmoji($change);
            $message .= "{$emoji} {$change['description']}\n";
        }

        $this->telegram_bot->sendMessage($message);
    }

    // نوتیفیکیشن سیگنال‌های تکنیکال
    private function notifyTechnicalSignals($token, $signals) {
        $message = "📊 <b>سیگنال تکنیکال برای {$token['symbol']}</b>\n\n";
        
        foreach ($signals['signals'] as $signal) {
            $message .= "• {$signal['description']}\n";
        }

        $this->telegram_bot->sendMessage($message);
    }

    // ارسال تحلیل روزانه
    private function sendDailyAnalysis() {
        $this->log("Preparing daily analysis...");
        
        try {
            // جمع‌آوری داده‌های روزانه
            $daily_data = $this->collectDailyData();
            
            // ارسال تحلیل
            $this->telegram_bot->sendDailyAnalysis($daily_data);
            
            // ذخیره زمان آخرین تحلیل
            $this->updateLastAnalysisTime();
            
            $this->log("Daily analysis sent successfully.");
        } catch (\Exception $e) {
            $this->log("Error sending daily analysis: " . $e->getMessage());
        }
    }

    // بررسی نیاز به ارسال تحلیل روزانه
    private function shouldSendDailyAnalysis() {
        $stmt = $this->db->prepare("
            SELECT value FROM settings 
            WHERE setting_key = 'last_daily_analysis'
        ");
        $stmt->execute();
        $last_analysis = $stmt->fetchColumn();

        return !$last_analysis || strtotime($last_analysis) < strtotime('today');
    }

    // به‌روزرسانی زمان آخرین تحلیل
    private function updateLastAnalysisTime() {
        $stmt = $this->db->prepare("
            INSERT INTO settings (setting_key, value) 
            VALUES ('last_daily_analysis', NOW())
            ON DUPLICATE KEY UPDATE value = NOW()
        ");
        $stmt->execute();
    }

    // لاگ کردن پیام‌ها
    private function log($message) {
        $log_message = "[" . date('Y-m-d H:i:s') . "] " . $message . "\n";
        file_put_contents($this->log_file, $log_message, FILE_APPEND);
    }

    // تبدیل تغییرات به ایموجی
    private function getChangeEmoji($change) {
        if ($change['type'] === 'price') {
            return $change['change'] > 0 ? '📈' : '📉';
        } elseif ($change['type'] === 'volume') {
            return '📊';
        } elseif ($change['type'] === 'holders') {
            return '👥';
        }
        return '➡️';
    }
}

// اجرای مانیتورینگ
try {
    $monitor = new Monitor();
    $monitor->run();
} catch (\Exception $e) {
    error_log("Critical error in monitor.php: " . $e->getMessage());
}